package org.msh.tb.reports2.variables;

import org.jboss.seam.international.Messages;
import org.msh.reports.filters.FilterOperation;
import org.msh.reports.filters.FilterOption;
import org.msh.reports.filters.ValueHandler;
import org.msh.reports.keys.Key;
import org.msh.reports.query.SQLDefs;
import org.msh.tb.entities.enums.MedicineLine;
import org.msh.tb.entities.enums.PatientType;
import org.msh.tb.reports2.VariableImpl;

import java.util.ArrayList;
import java.util.List;

public class TypeTBCaseVariable extends VariableImpl {

    private static final String KEY_NEW = "anew";
    private static final String KEY_1LINE = "line1";
    private static final String KEY_2LINE = "line2";


    public TypeTBCaseVariable() {
        super("tb_hist", "manag.reportgen.var.tbcasetype", null);
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#prepareVariableQuery(org.msh.reports.query.SQLDefs, int)
     */
    @Override
    public void prepareVariableQuery(SQLDefs def, int iteration) {
        String key;
        switch (iteration) {
            // NEW PATIENT
            case 0:
                key = KEY_NEW;
                break;

            // PREVIOUSLY TREATED WITH 1ST LINE DRUGS
            case 1:
                key = KEY_1LINE;
                break;

            // PREVIOUSLY TREATED WITH 2ND LINE DRUGS
            case 2:
                key = KEY_2LINE;
                break;

            default:
                throw new IllegalArgumentException("Iteration not expected: " + iteration);
        }

        def.select("'" + key + "'");
        def.addRestriction(createRestrictions(key));
    }


    /**
     * Add common restrictions to variables and filters
     * @param key
     */
    protected String createRestrictions(String key) {
        // NEW PATIENT
        if (KEY_NEW.equals(key)) {
            return "tbcase.patientType = " + PatientType.NEW.ordinal();
        }

        // PREVIOUSLY TREATED WITH 1ST LINE DRUGS
        if (KEY_1LINE.equals(key)) {
            return "tbcase.patientType != " + PatientType.NEW.ordinal() +
                    " and not exists(select * from prevtbtreatment pv " +
                    "inner join res_prevtbtreatment d1 on d1.prevtbtreatment_id = pv.id " +
                    "inner join substance c1 on c1.id = d1.substance_id " +
                    "where c1.line = " + MedicineLine.SECOND_LINE.ordinal() + " and pv.case_id = tbcase.id)";
        }

        // PREVIOUSLY TREATED WITH 2ND LINE DRUGS
        if (KEY_2LINE.equals(key)) {
            return "tbcase.patientType <> " + PatientType.NEW.ordinal() +
                    " and exists(select * from prevtbtreatment pv " +
                    "inner join res_prevtbtreatment d1 on d1.prevtbtreatment_id = pv.id " +
                    "inner join substance c1 on c1.id = d1.substance_id " +
                    "where c1.line = " + MedicineLine.SECOND_LINE.ordinal() + " and pv.case_id = tbcase.id)";
        }

        throw new RuntimeException("Invalid key " + key);
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#prepareFilterQuery(org.msh.reports.query.SQLDefs, org.msh.reports.filters.FilterOperation, java.lang.Object)
     */
    @Override
    public void prepareFilterQuery(SQLDefs def, FilterOperation oper, ValueHandler value) {
        String sql = value.mapSqlOR(new ValueHandler.ValueIterator() {
            @Override
            public String iterate(String value, int index) {
                return createRestrictions(value);
            }
        });
        def.addRestriction(sql);
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#createKey(java.lang.Object)
     */
    @Override
    public Key createKey(Object values, int iteration) {
        return super.createKey(values, iteration);
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#getIteractionCount()
     */
    @Override
    public int getIteractionCount() {
        return 3;
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#filterValueFromString(java.lang.String)
     */
    @Override
    public Object filterValueFromString(String value) {
        if ((value == null) || (!((KEY_NEW.equals(value)) || (KEY_1LINE.equals(value)) || (KEY_2LINE.equals(value))) ))
            throw new IllegalArgumentException("Wrong type for filter value: " + value);

        return value;
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#getDisplayText(java.lang.Object)
     */
    @Override
    public String getDisplayText(Key key) {
        Object val = key.getValue();

        if (KEY_NEW.equals(val)) {
            return Messages.instance().get("manag.confmdrrep.new");
        }

        if (KEY_1LINE.equals(val)) {
            return Messages.instance().get("manag.confmdrrep.prev12line");
        }

        if (KEY_2LINE.equals(val)) {
            return Messages.instance().get("manag.confmdrrep.prev1line");
        }

        return super.getDisplayText(key);
    }

    /* (non-Javadoc)
     * @see org.msh.tb.reports2.VariableImpl#getFilterOptions(java.lang.Object)
     */
    @Override
    public List<FilterOption> getFilterOptions(Object param) {
        List<FilterOption> lst = new ArrayList<FilterOption>();
        lst.add(createFilterOption(KEY_NEW));
        lst.add(createFilterOption(KEY_1LINE));
        lst.add(createFilterOption(KEY_2LINE));

        return lst;
    }

}
